﻿# Copyright (c) 2001-2017 WorldViz LLC.
# All rights reserved.
""" This is a simple implementation of a "fly to" transport.

The fly to transport works like a grappling hook, a ray is shot from the transport's
location and attaches to a location indicated by the ray caster node. The ray caster
node provides orientation and a virtual line is intersected with the scene to get the
attachment point.
"""

import viz
import vizmat
import vizshape

from transportation import Transport


class GrapplingHook(Transport):
	"""Creates a new instance of the fly to transport.
	"""
	def __init__(self, movable, rayCaster=None, pivot=None, debug=False):
		Transport.__init__(self, node=movable, pivot=pivot, debug=debug)
		
		if rayCaster is None:
			rayCaster = viz.addGroup()
		self._rayCaster = rayCaster
		
		self._movementVelocity = 0.03
		
		self._fired = False
		self._blocked = False
		
		self.indicatorNode = viz.addGroup()
		self.targetLocation = [0, 0, 0]
		self.pendingIntersection = False
		
		self._offsetShift = [0, 0, 0.5]
		
		self._makeRay()
		self._makeSnapIndicator()
	
	def _getLookAt(self, intersection):
		"""Internal function that returns the appropriate transform for a node"""
		mat = vizmat.Transform()
		target = vizmat.Vector(intersection.point)+vizmat.Vector(intersection.normal)
		if intersection.normal == [0, 1, 0]:
			mat.makeLookAt(intersection.point, target, [0, 0, 1])
		elif intersection.normal == [0, -1, 0]:
			mat.makeLookAt(intersection.point, target, [0, 0, 1])
		else:
			mat.makeLookAt(intersection.point, target, [0, 1, 0])
		return mat
	
	def _assignLookAt(self, intersection, node):
		"""Internal function used to make a node face the appropriate direction."""
		mat = self._getLookAt(intersection)
		node.setQuat(mat.getQuat())
		node.setPosition(intersection.point, viz.ABS_GLOBAL)
	
	def clear(self, mag=1):
		"""Clears the target indicator"""
		self.targetIndicator.visible(False)
		self._rayCaster.getRay().visible(False)
	
	def showIntersection(self, mag=1):
		"""Shows the intersection"""
		self.clear()
		self.pendingIntersection = True
	
	def _intersect(self):
		"""Internal function that handles the intersections"""
		self._ray.visible(False)
		
		if self._pivot is None:
			p1 = self.getPosition(viz.ABS_GLOBAL)
		else:
			p1 = self._pivot.getPosition(viz.ABS_GLOBAL)
		p2 = viz.vertex(obj.point)
		
		self._ray.setVertex(0, p1)
		self._ray.setVertex(1, p2)
		
		# get the orientation from the ray caster node
		mat = self.rayCaster.getMatrix(viz.ABS_GLOBAL)
		
		targetPos = vizmat.Vector(mat.preMultVec([0, 0, 1000]))
		offsetStartingPos = vizmat.Vector(mat.preMultVec(self._offsetShift))
		
		obj = viz.intersect(offsetStartingPos, targetPos, ignoreBackFace=True)
		
		return obj
	
	def flyTo(self, e=None):
		"""Starts the user moving toward the selected destination
		"""
		if self.targetIndicator.getVisible():
			self.targetLocation = vizmat.Vector(self.targetIndicator.getPosition(viz.ABS_GLOBAL))
	
	def finalize(self, e=None):
		"""Finalizes pending operations"""
		if self.targetLocation is not None:
			currentPos = vizmat.Vector(self.getPosition(viz.ABS_GLOBAL))
			# adjust for the pivot
			if self._pivot is None:
				vec = vizmat.Vector(self.targetLocation - currentPos)
			else:
				vec = vizmat.Vector(self.targetLocation - vizmat.Vector(self._pivot.getPosition(viz.ABS_GLOBAL)))
			length = vec.length()
			if length > 0.001:
				vec.normalize()
				vec = vec*min(length, self._movementVelocity)
				self.setPosition(currentPos + vec, viz.ABS_GLOBAL)
		self.targetLocation = None
	
	def _makeSnapIndicator(self):
		"""Internal function that makes snap to indicator"""
		plane = vizshape.addPlane((.25, .25), axis=vizshape.AXIS_Z)
		plane.color(0, .5, 0)
		plane.alpha(.7)
		plane.disable(viz.LIGHTING)
		plane.setPosition(0, .1, 0)
		frag_outline = """
				void main()
				{
				gl_FragColor = vec4(0, 1, 0, 1);
				}
				"""
		shader_outline = viz.addShader(frag=frag_outline)
		plane.wire = plane.clone(parent=plane, opMode=viz.OP_OVERRIDE|viz.OP_ROOT)
		plane.wire.polyMode(viz.POLY_WIRE)
		plane.wire.lineWidth(2)
		plane.wire.apply(shader_outline)
		plane.wire.stencilFunc(viz.StencilFunc.RenderMask(), op=viz.OP_ROOT)
		plane.stencilFunc(viz.StencilFunc.RenderOutsideMask(), op=viz.OP_ROOT)
		plane.disable(viz.INTERSECTION)
		plane.disable(viz.DEPTH_TEST)
		plane.drawOrder(10000)
		plane.visible(0)
		self._targetIndicator = plane
	
	def _makeRay(self):
		"""Internal function that makes the ray"""
		viz.startLayer(viz.LINES)
		viz.lineWidth(3)
		viz.pointSize(15)
		viz.vertexColor(1, 1, 0)
		viz.vertex([0, 0, 0])
		viz.vertexColor(0, 1, 0)
		viz.vertex([0, 0, 0])
		self._ray = viz.endLayer()



